<?php
/**
 * Cache Management Interface
 */

session_start();

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once '../config/database.php';
require_once '../api/cache_manager.php';
require_once 'includes/helpers.php';

$db = new Database();
$conn = $db->getConnection();

$cacheManager = new CacheManager($conn);

// User-Daten
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.full_name, u.username, u.role
        FROM admin_users u
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
        $current_user_role = $user_data['role'] ?? 'admin';
    }
}

// Theme
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'clear_cache') {
    $autohausId = $_POST['autohaus_id'] ?? null;
    if ($autohausId) {
        $cacheManager->invalidateCache($autohausId);
        $message = "Cache für Autohaus $autohausId wurde geleert.";
    } else {
        // Clear all cache
        $cacheManager->cleanupExpiredCache();
        $message = "Alle abgelaufenen Cache-Einträge wurden bereinigt.";
    }
}

$cacheStats = $cacheManager->getCacheStats();
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cache Management - Boxenstop Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="assets/css/admin-zoom.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/glassmorphism-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-main {
            margin-left: 300px;
            transition: margin-left 0.3s ease;
            width: calc(100% - 300px);
            padding: 2rem;
            min-height: 100vh;
        }

        .smart-sidebar.collapsed + .admin-main,
        .figma-sidebar.collapsed ~ .admin-main {
            margin-left: 80px;
            width: calc(100% - 80px);
        }

        .cache-page-header {
            margin-bottom: 2rem;
        }

        .cache-page-header h1 {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .cache-page-header p {
            color: #64748b;
            font-size: 1rem;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: #ffffff;
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            display: flex;
            align-items: center;
            gap: 1.25rem;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .stat-icon {
            width: 56px;
            height: 56px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
            color: #ffffff;
        }

        .stat-icon.fa-clock {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        }

        .stat-icon.fa-calendar-alt {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        }

        .stat-content {
            flex: 1;
        }

        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            line-height: 1.2;
            margin-bottom: 0.25rem;
        }

        .stat-label {
            font-size: 0.875rem;
            color: #64748b;
            font-weight: 500;
        }

        .action-buttons {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            border: none;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            transition: all 0.2s;
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .btn-warning {
            background: #f59e0b;
            color: #ffffff;
        }

        .btn-warning:hover {
            background: #d97706;
        }

        .btn-info {
            background: #06b6d4;
            color: #ffffff;
        }

        .btn-info:hover {
            background: #0891b2;
        }

        .btn-success {
            background: #10b981;
            color: #ffffff;
        }

        .btn-success:hover {
            background: #059669;
        }

        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 1rem;
        }

        .info-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 1rem;
            background: #f8fafc;
            border-radius: 8px;
            border: 1px solid #e2e8f0;
        }

        .info-item strong {
            color: #1e293b;
            font-weight: 600;
        }

        .info-item span {
            color: #64748b;
            font-size: 0.875rem;
        }

        .alert {
            padding: 1rem 1.5rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .alert-success {
            background: #d1fae5;
            color: #065f46;
            border: 1px solid #6ee7b7;
        }

        .card {
            background: #ffffff;
            border-radius: 12px;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
            overflow: hidden;
        }

        .card-header {
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            background: #f8fafc;
        }

        .card-header h2 {
            font-size: 1.25rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .card-body {
            padding: 1.5rem;
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <div class="cache-page-header">
                <h1>
                    <i class="fas fa-database"></i>
                    Cache Management
                </h1>
                <p>Verwalten Sie den System-Cache für optimale Performance</p>
            </div>

            <?php if (isset($message)): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle"></i>
                    <?= htmlspecialchars($message) ?>
                </div>
            <?php endif; ?>

            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-database"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number"><?= $cacheStats['total_entries'] ?? 0 ?></div>
                        <div class="stat-label">Gesamte Einträge</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon fa-clock">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number"><?= $cacheStats['recent_entries'] ?? 0 ?></div>
                        <div class="stat-label">Aktuelle Einträge</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon fa-calendar-alt">
                        <i class="fas fa-calendar-alt"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number">
                            <?php 
                            if (isset($cacheStats['newest_entry']) && $cacheStats['newest_entry']) {
                                echo date('H:i', strtotime($cacheStats['newest_entry']));
                            } else {
                                echo '--:--';
                            }
                            ?>
                        </div>
                        <div class="stat-label">Letzter Eintrag</div>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-tools"></i> Cache Aktionen</h2>
                </div>
                <div class="card-body">
                    <div class="action-buttons">
                        <button class="btn btn-warning" onclick="clearAllCache()">
                            <i class="fas fa-trash"></i>
                            Alle Caches leeren
                        </button>
                        
                        <button class="btn btn-info" onclick="cleanupCache()">
                            <i class="fas fa-broom"></i>
                            Abgelaufene Einträge bereinigen
                        </button>
                        
                        <button class="btn btn-success" onclick="refreshStats()">
                            <i class="fas fa-sync"></i>
                            Statistiken aktualisieren
                        </button>
                    </div>
                </div>
            </div>

            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-info-circle"></i> Cache Informationen</h2>
                </div>
                <div class="card-body">
                    <div class="info-grid">
                        <div class="info-item">
                            <strong>Datenbank-Cache:</strong>
                            <span>15 Minuten TTL</span>
                        </div>
                        <div class="info-item">
                            <strong>HTML-Cache:</strong>
                            <span>1 Stunde TTL</span>
                        </div>
                        <div class="info-item">
                            <strong>Cache-Verzeichnis:</strong>
                            <span>/cache/</span>
                        </div>
                        <div class="info-item">
                            <strong>Automatische Bereinigung:</strong>
                            <span>Bei jedem Aufruf</span>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        function clearAllCache() {
            if (confirm('Möchten Sie wirklich alle Caches leeren?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="action" value="clear_cache">';
                document.body.appendChild(form);
                form.submit();
            }
        }

        function cleanupCache() {
            if (confirm('Möchten Sie abgelaufene Cache-Einträge bereinigen?')) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.innerHTML = '<input type="hidden" name="action" value="clear_cache">';
                document.body.appendChild(form);
                form.submit();
            }
        }

        function refreshStats() {
            location.reload();
        }
    </script>
</body>
</html>
