<?php
/**
 * Audit Log - Übersicht gelöschter Termine
 */

session_start();

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: login.php');
    exit;
}

require_once '../config/database.php';
require_once 'includes/helpers.php';

$db = new Database();
$conn = $db->getConnection();

// User-Daten
$current_user_id = $_SESSION['admin_user_id'] ?? null;
$current_user_name = 'Admin';
$current_user_role = 'admin';

if ($current_user_id) {
    $stmt = $conn->prepare("
        SELECT u.full_name, u.username, u.role
        FROM admin_users u
        WHERE u.id = ?
    ");
    $stmt->execute([$current_user_id]);
    $user_data = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($user_data) {
        $current_user_name = $user_data['full_name'] ?? $user_data['username'] ?? 'Benutzer';
        $current_user_role = $user_data['role'] ?? 'admin';
    }
}

// Theme
$current_theme = 'light';
if ($current_user_id) {
    $stmt = $conn->prepare("SELECT theme FROM admin_users WHERE id = ?");
    $stmt->execute([$current_user_id]);
    $theme_result = $stmt->fetch();
    $current_theme = $theme_result['theme'] ?? 'light';
}

// Get filter parameters
$date_from = $_GET['date_from'] ?? date('Y-m-01'); // First day of current month
$date_to = $_GET['date_to'] ?? date('Y-m-d');
$user_filter = $_GET['user'] ?? '';
$search = $_GET['search'] ?? '';

// Check if audit table exists
$table_check = $conn->query("SHOW TABLES LIKE 'deleted_appointments_audit'");
if ($table_check->rowCount() == 0) {
    // Table doesn't exist, redirect to setup
    header('Location: ../setup_audit_tables.php');
    exit;
}

// Build query
$where_conditions = ["deleted_at BETWEEN ? AND ?"];
$params = [$date_from . ' 00:00:00', $date_to . ' 23:59:59'];

if (!empty($user_filter)) {
    $where_conditions[] = "deleted_by_username = ?";
    $params[] = $user_filter;
}

if (!empty($search)) {
    $where_conditions[] = "(customer_name LIKE ? OR license_plate LIKE ? OR storage_number LIKE ?)";
    $search_param = "%$search%";
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
}

$where_clause = implode(' AND ', $where_conditions);

// Get deleted appointments
$sql = "
    SELECT * FROM deleted_appointments_audit 
    WHERE $where_clause 
    ORDER BY deleted_at DESC
    LIMIT 100
";

$stmt = $conn->prepare($sql);
$stmt->execute($params);
$deleted_appointments = $stmt->fetchAll();

// Get unique users for filter
$user_stmt = $conn->prepare("SELECT DISTINCT deleted_by_username FROM deleted_appointments_audit ORDER BY deleted_by_username");
$user_stmt->execute();
$users = $user_stmt->fetchAll(PDO::FETCH_COLUMN);

// Get statistics
$stats_sql = "
    SELECT 
        COUNT(*) as total_deleted,
        COUNT(CASE WHEN tire_cancelled = 1 THEN 1 END) as tire_cancelled_count,
        COUNT(DISTINCT deleted_by_username) as unique_users
    FROM deleted_appointments_audit 
    WHERE $where_clause
";

$stats_stmt = $conn->prepare($stats_sql);
$stats_stmt->execute($params);
$stats = $stats_stmt->fetch();
?>
<!DOCTYPE html>
<html lang="de" data-theme="<?php echo htmlspecialchars($current_theme); ?>">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Audit Log - Gelöschte Termine</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="assets/css/admin.css">
    <link rel="stylesheet" href="<?php echo asset_url('assets/css/unified-sidebar.css'); ?>">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .admin-main {
            margin-left: 280px;
            transition: margin-left 0.3s ease;
            width: calc(100% - 280px);
            padding: 2rem;
            min-height: 100vh;
        }

        .new-sidebar.collapsed ~ .admin-main,
        .admin-main.sidebar-collapsed {
            margin-left: 80px;
            width: calc(100% - 80px);
        }

        .audit-page-header {
            margin-bottom: 2rem;
        }

        .audit-page-header h1 {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 0.5rem;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .audit-page-header p {
            color: #64748b;
            font-size: 1rem;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
            margin-bottom: 2rem;
        }

        .stat-card {
            background: #ffffff;
            border-radius: 12px;
            padding: 1.5rem;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            display: flex;
            align-items: center;
            gap: 1.25rem;
            transition: transform 0.2s, box-shadow 0.2s;
        }

        .stat-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .stat-icon {
            width: 56px;
            height: 56px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
            color: #ffffff;
        }

        .stat-icon.fa-tire {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        }

        .stat-icon.fa-users {
            background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
        }

        .stat-content {
            flex: 1;
        }

        .stat-number {
            font-size: 2rem;
            font-weight: 700;
            color: #1e293b;
            line-height: 1.2;
            margin-bottom: 0.25rem;
        }

        .stat-label {
            font-size: 0.875rem;
            color: #64748b;
            font-weight: 500;
        }

        .filters {
            background: #ffffff;
            border-radius: 12px;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            padding: 1.5rem;
            margin-bottom: 2rem;
        }

        .filter-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
            align-items: end;
        }

        .filter-group {
            display: flex;
            flex-direction: column;
        }

        .filter-group label {
            font-weight: 500;
            color: #374151;
            margin-bottom: 0.5rem;
            font-size: 0.875rem;
        }

        .filter-group input,
        .filter-group select {
            padding: 0.75rem;
            border: 1px solid #d1d5db;
            border-radius: 8px;
            font-size: 0.875rem;
            background: #ffffff;
            color: #1e293b;
            transition: border-color 0.2s;
        }

        .filter-group input:focus,
        .filter-group select:focus {
            outline: none;
            border-color: #3b82f6;
        }

        .btn {
            display: inline-flex;
            align-items: center;
            gap: 0.5rem;
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 8px;
            font-weight: 600;
            font-size: 0.875rem;
            cursor: pointer;
            transition: all 0.2s;
            text-decoration: none;
        }

        .btn:hover {
            transform: translateY(-1px);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
        }

        .btn-primary {
            background: #3b82f6;
            color: #ffffff;
        }

        .btn-primary:hover {
            background: #2563eb;
        }

        .btn-secondary {
            background: #6b7280;
            color: #ffffff;
        }

        .btn-secondary:hover {
            background: #4b5563;
        }

        .card {
            background: #ffffff;
            border-radius: 12px;
            border: 1px solid #e2e8f0;
            box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1);
            margin-bottom: 2rem;
            overflow: hidden;
        }

        .card-header {
            padding: 1.5rem;
            border-bottom: 1px solid #e2e8f0;
            background: #f8fafc;
        }

        .card-header h2 {
            font-size: 1.25rem;
            font-weight: 600;
            color: #1e293b;
            margin: 0;
            display: flex;
            align-items: center;
            gap: 0.75rem;
        }

        .card-body {
            padding: 0;
        }

        .audit-table {
            width: 100%;
            border-collapse: collapse;
        }

        .audit-table th {
            background: #f1f5f9;
            padding: 1rem;
            text-align: left;
            font-weight: 600;
            color: #374151;
            border-bottom: 1px solid #e5e7eb;
            font-size: 0.875rem;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .audit-table td {
            padding: 1rem;
            border-bottom: 1px solid #f3f4f6;
            vertical-align: top;
            font-size: 0.875rem;
        }

        .audit-table tr:hover {
            background: #f8fafc;
        }

        .customer-info {
            font-weight: 600;
            color: #1e293b;
            margin-bottom: 0.25rem;
        }

        .license-plate {
            background: #3b82f6;
            color: white;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 600;
            display: inline-block;
        }

        .deletion-info {
            font-size: 0.8125rem;
            color: #6b7280;
        }

        .deletion-reason {
            max-width: 200px;
            overflow: hidden;
            text-overflow: ellipsis;
            white-space: nowrap;
        }

        .tire-status {
            display: inline-flex;
            align-items: center;
            gap: 0.25rem;
            padding: 0.25rem 0.5rem;
            border-radius: 4px;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .tire-status.cancelled {
            background: #d1fae5;
            color: #065f46;
        }

        .tire-status.not-cancelled {
            background: #fef3c7;
            color: #92400e;
        }

        .no-data {
            text-align: center;
            padding: 3rem;
            color: #6b7280;
        }

        .no-data i {
            font-size: 3rem;
            margin-bottom: 1rem;
            opacity: 0.5;
            color: #9ca3af;
        }

        .no-data h3 {
            font-size: 1.25rem;
            font-weight: 600;
            color: #374151;
            margin-bottom: 0.5rem;
        }

        .no-data p {
            color: #6b7280;
        }

        .action-buttons {
            display: flex;
            gap: 1rem;
            justify-content: center;
            margin-top: 2rem;
        }

        @media (max-width: 768px) {
            .admin-main {
                margin-left: 0 !important;
                width: 100% !important;
                padding: 1rem;
            }

            .filter-row {
                grid-template-columns: 1fr;
            }

            .stats-grid {
                grid-template-columns: 1fr;
            }

            .card-body {
                overflow-x: auto;
            }

            .audit-table {
                min-width: 800px;
            }
        }
    </style>
</head>
<body class="admin-body">
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>

        <main class="admin-main">
            <div class="audit-page-header">
                <h1>
                    <i class="fas fa-history"></i>
                    Audit Log - Gelöschte Termine
                </h1>
                <p>Vollständige Nachverfolgung aller gelöschten Termine und Reifen-Stornierungen</p>
            </div>

            <!-- Statistics -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-trash"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number"><?php echo $stats['total_deleted']; ?></div>
                        <div class="stat-label">Gelöschte Termine</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon fa-tire">
                        <i class="fas fa-tire"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number"><?php echo $stats['tire_cancelled_count']; ?></div>
                        <div class="stat-label">Reifen storniert</div>
                    </div>
                </div>
                <div class="stat-card">
                    <div class="stat-icon fa-users">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number"><?php echo $stats['unique_users']; ?></div>
                        <div class="stat-label">Aktive Benutzer</div>
                    </div>
                </div>
            </div>

            <!-- Filters -->
            <div class="filters">
                <form method="GET" class="filter-form">
                    <div class="filter-row">
                        <div class="filter-group">
                            <label for="date_from">Von Datum:</label>
                            <input type="date" id="date_from" name="date_from" value="<?php echo htmlspecialchars($date_from); ?>">
                        </div>
                        <div class="filter-group">
                            <label for="date_to">Bis Datum:</label>
                            <input type="date" id="date_to" name="date_to" value="<?php echo htmlspecialchars($date_to); ?>">
                        </div>
                        <div class="filter-group">
                            <label for="user">Benutzer:</label>
                            <select id="user" name="user">
                                <option value="">Alle Benutzer</option>
                                <?php foreach ($users as $user): ?>
                                    <option value="<?php echo htmlspecialchars($user); ?>" <?php echo $user_filter === $user ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($user); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="filter-group">
                            <label for="search">Suche:</label>
                            <input type="text" id="search" name="search" value="<?php echo htmlspecialchars($search); ?>" placeholder="Kunde, Kennzeichen, Einlagerungsnummer">
                        </div>
                        <div class="filter-group">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-search"></i>
                                Filtern
                            </button>
                            <a href="audit_log.php" class="btn btn-secondary" style="margin-top: 0.5rem;">
                                <i class="fas fa-times"></i>
                                Zurücksetzen
                            </a>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Audit Table -->
            <div class="card">
                <div class="card-header">
                    <h2>
                        <i class="fas fa-list"></i>
                        Gelöschte Termine (<?php echo count($deleted_appointments); ?> Einträge)
                    </h2>
                </div>
                <div class="card-body">
                    <?php if (empty($deleted_appointments)): ?>
                        <div class="no-data">
                            <i class="fas fa-inbox"></i>
                            <h3>Keine gelöschten Termine gefunden</h3>
                            <p>Für den gewählten Zeitraum wurden keine gelöschten Termine gefunden.</p>
                        </div>
                    <?php else: ?>
                        <table class="audit-table">
                            <thead>
                                <tr>
                                    <th>Kunde</th>
                                    <th>Event</th>
                                    <th>Datum/Zeit</th>
                                    <th>Einlagerung</th>
                                    <th>Reifen-Status</th>
                                    <th>Gelöscht von</th>
                                    <th>Grund</th>
                                    <th>Zeitstempel</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($deleted_appointments as $appointment): ?>
                                    <tr>
                                        <td>
                                            <div class="customer-info"><?php echo htmlspecialchars($appointment['customer_name']); ?></div>
                                            <div class="license-plate"><?php echo htmlspecialchars($appointment['license_plate']); ?></div>
                                        </td>
                                        <td><?php echo htmlspecialchars($appointment['event_name']); ?></td>
                                        <td>
                                            <div><?php echo date('d.m.Y', strtotime($appointment['slot_date'])); ?></div>
                                            <div class="deletion-info"><?php echo date('H:i', strtotime($appointment['slot_time'])); ?></div>
                                        </td>
                                        <td>
                                            <?php if ($appointment['storage_number']): ?>
                                                <div><?php echo htmlspecialchars($appointment['storage_number']); ?></div>
                                                <?php if ($appointment['tire_order_date']): ?>
                                                    <div class="deletion-info">Bestellt: <?php echo date('d.m.Y H:i', strtotime($appointment['tire_order_date'])); ?></div>
                                                <?php endif; ?>
                                            <?php else: ?>
                                                <span class="deletion-info">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if ($appointment['storage_number'] && $appointment['tire_order_date']): ?>
                                                <span class="tire-status <?php echo $appointment['tire_cancelled'] ? 'cancelled' : 'not-cancelled'; ?>">
                                                    <i class="fas fa-tire"></i>
                                                    <?php echo $appointment['tire_cancelled'] ? 'Storniert' : 'Nicht storniert'; ?>
                                                </span>
                                            <?php else: ?>
                                                <span class="deletion-info">-</span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <div><?php echo htmlspecialchars($appointment['deleted_by_username']); ?></div>
                                            <div class="deletion-info">ID: <?php echo $appointment['deleted_by_user_id']; ?></div>
                                        </td>
                                        <td>
                                            <div class="deletion-reason" title="<?php echo htmlspecialchars($appointment['deletion_reason']); ?>">
                                                <?php echo htmlspecialchars($appointment['deletion_reason'] ?: '-'); ?>
                                            </div>
                                        </td>
                                        <td>
                                            <div><?php echo date('d.m.Y', strtotime($appointment['deleted_at'])); ?></div>
                                            <div class="deletion-info"><?php echo date('H:i:s', strtotime($appointment['deleted_at'])); ?></div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    <?php endif; ?>
                </div>
            </div>

            <!-- Navigation -->
            <div class="action-buttons">
                <a href="appointments.php" class="btn btn-primary">
                    <i class="fas fa-arrow-left"></i>
                    Zurück zu Terminen
                </a>
            </div>
        </main>
    </div>
</body>
</html>
