// Theme Management System for Admin Panel
class ThemeManager {
    constructor() {
        this.init();
    }
    
    init() {
        // Apply theme on page load
        this.applyTheme();
        
        // Listen for system theme changes
        this.setupSystemThemeListener();
    }
    
    applyTheme(theme = null) {
        const body = document.body;
        const currentTheme = theme || this.getCurrentTheme();
        
        // Remove existing theme classes
        body.classList.remove('dark-theme', 'light-theme', 'auto-theme');
        
        if (currentTheme === 'dark') {
            body.classList.add('dark-theme');
        } else if (currentTheme === 'light') {
            body.classList.add('light-theme');
        } else if (currentTheme === 'auto') {
            // Auto theme - use system preference
            if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
                body.classList.add('dark-theme');
            } else {
                body.classList.add('light-theme');
            }
        }
    }
    
    getCurrentTheme() {
        // Check if theme is stored in localStorage
        const storedTheme = localStorage.getItem('admin_theme');
        if (storedTheme) {
            return storedTheme;
        }
        
        // Check if theme is set in the form
        const themeSelect = document.getElementById('admin_theme');
        if (themeSelect) {
            return themeSelect.value;
        }
        
        // Check if theme is set in data-theme attribute
        const body = document.body;
        const dataTheme = body.getAttribute('data-theme');
        if (dataTheme) {
            return dataTheme;
        }
        
        // Default to light theme
        return 'light';
    }
    
    saveThemePreference(theme) {
        // Save to localStorage for immediate effect
        localStorage.setItem('admin_theme', theme);
        
        // Also save to server via AJAX
        fetch('save_theme_preference.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify({
                theme: theme
            })
        }).catch(error => {
            console.log('Theme preference saved locally, server sync failed:', error);
        });
    }
    
    setupSystemThemeListener() {
        // Listen for system theme changes when auto theme is selected
        if (window.matchMedia) {
            window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', (e) => {
                const currentTheme = this.getCurrentTheme();
                if (currentTheme === 'auto') {
                    this.applyTheme('auto');
                }
            });
        }
    }
    
    // Public method to change theme
    setTheme(theme) {
        this.applyTheme(theme);
        this.saveThemePreference(theme);
    }
    
    // Public method to get current theme
    getTheme() {
        return this.getCurrentTheme();
    }
}

// Initialize theme manager when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    window.themeManager = new ThemeManager();
    
    // Setup theme change handler if theme select exists
    const themeSelect = document.getElementById('admin_theme');
    if (themeSelect) {
        themeSelect.addEventListener('change', function() {
            const selectedTheme = this.value;
            window.themeManager.setTheme(selectedTheme);
        });
    }
});

// Export for use in other scripts
if (typeof module !== 'undefined' && module.exports) {
    module.exports = ThemeManager;
}
