<?php
/**
 * AJAX-Endpoint für einzelne Tests
 * Führt einen einzelnen Test aus und gibt das Ergebnis zurück
 */

// Fehler-Reporting deaktivieren für sauberes JSON
error_reporting(0);
ini_set('display_errors', 0);

session_start();

// Output-Buffering starten um sicherzustellen, dass nur JSON ausgegeben wird
ob_start();

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-cache, must-revalidate');

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    ob_clean();
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

$test_key = $_POST['test_key'] ?? $_GET['test_key'] ?? '';

if (empty($test_key)) {
    echo json_encode(['success' => false, 'error' => 'Kein Test angegeben']);
    exit;
}

// Lade benötigte Dateien
if (file_exists('../config/database_optimized.php')) {
    require_once '../config/database_optimized.php';
    $db = DatabaseOptimized::getInstance();
    $conn = $db->getConnection();
} else {
    require_once '../config/database.php';
    $db = new Database();
    $conn = $db->getConnection();
}

$start = microtime(true);
$result = null;

try {
    switch ($test_key) {
        case 'database_connection':
            try {
                if (method_exists($db, 'fetchColumn')) {
                    $db->fetchColumn("SELECT COUNT(*) FROM admin_users WHERE role = 'admin'");
                } else {
                    $stmt = $conn->prepare("SELECT COUNT(*) FROM admin_users WHERE role = 'admin'");
                    $stmt->execute();
                    $stmt->fetchColumn();
                }
                $db_time = (microtime(true) - $start) * 1000;
                $result = [
                    'name' => 'Datenbankverbindung',
                    'status' => 'success',
                    'time' => $db_time,
                    'score' => $db_time < 10 ? 100 : ($db_time < 50 ? 80 : 60),
                    'details' => 'PDO-Verbindung erfolgreich hergestellt',
                    'color' => $db_time < 10 ? 'success' : ($db_time < 50 ? 'warning' : 'error')
                ];
            } catch (Exception $e) {
                $result = [
                    'name' => 'Datenbankverbindung',
                    'status' => 'error',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => 0,
                    'details' => 'Fehler: ' . $e->getMessage(),
                    'color' => 'error'
                ];
            }
            break;
            
        case 'backup_system':
            // Erstelle Backup-Verzeichnis falls nicht vorhanden
            $backup_dir = '../../backups';
            if (!is_dir($backup_dir)) {
                @mkdir($backup_dir, 0755, true);
            }
            
            if (file_exists('../../api/backup_system.php')) {
                require_once '../../api/backup_system.php';
                $backupSystem = new BackupSystem($conn);
                $available = $backupSystem->isAvailable();
                $backups = $backupSystem->listBackups();
                $result = [
                    'name' => 'Backup-System',
                    'status' => $available ? 'success' : 'warning',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => $available ? 100 : 50,
                    'details' => $available ? "Backup-System verfügbar (" . count($backups) . " Backups)" : 'Backup-Verzeichnis nicht verfügbar',
                    'color' => $available ? 'success' : 'warning'
                ];
            } else {
                $backup_dir_exists = is_dir($backup_dir);
                $backup_dir_writable = $backup_dir_exists && is_writable($backup_dir);
                $result = [
                    'name' => 'Backup-System',
                    'status' => $backup_dir_writable ? 'success' : 'warning',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => $backup_dir_writable ? 100 : 0,
                    'details' => $backup_dir_writable ? 'Backup-Verzeichnis verfügbar' : 'Backup-Verzeichnis nicht verfügbar',
                    'color' => $backup_dir_writable ? 'success' : 'warning'
                ];
            }
            break;
            
        case 'email_system':
            // Versuche Tabellen zu erstellen falls nicht vorhanden
            try {
                $conn->exec("CREATE TABLE IF NOT EXISTS email_templates (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    template_key VARCHAR(100) UNIQUE NOT NULL,
                    subject VARCHAR(255) NOT NULL,
                    body TEXT NOT NULL,
                    is_active BOOLEAN DEFAULT TRUE,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
                )");
            } catch (Exception $e) {}
            
            try {
                $conn->exec("CREATE TABLE IF NOT EXISTS email_logs (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    appointment_id INT NULL,
                    template_key VARCHAR(100) NULL,
                    recipient_email VARCHAR(255) NOT NULL,
                    subject VARCHAR(255) NOT NULL,
                    status ENUM('sent', 'failed', 'pending') DEFAULT 'pending',
                    sent_at DATETIME NULL,
                    error_message TEXT NULL,
                    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
                )");
            } catch (Exception $e) {}
            
            $email_templates_table = false;
            $email_logs_table = false;
            try {
                $stmt = $conn->query("SELECT COUNT(*) FROM email_templates LIMIT 1");
                $email_templates_table = true;
            } catch (Exception $e) {}
            try {
                $stmt = $conn->query("SELECT COUNT(*) FROM email_logs LIMIT 1");
                $email_logs_table = true;
            } catch (Exception $e) {}
            $score = 0;
            if ($email_templates_table) $score += 50;
            if ($email_logs_table) $score += 50;
            $result = [
                'name' => 'E-Mail-System',
                'status' => $score >= 50 ? 'success' : 'warning',
                'time' => (microtime(true) - $start) * 1000,
                'score' => $score,
                'details' => ($email_templates_table && $email_logs_table) ? 'E-Mail-System vollständig konfiguriert' : 'E-Mail-Tabellen fehlen teilweise',
                'color' => $score >= 50 ? 'success' : 'warning'
            ];
            break;
            
        case 'session_config':
            if (file_exists('../../api/session_handler.php')) {
                require_once '../../api/session_handler.php';
                $checks = SecureSessionHandler::isSecure();
                $checks_passed = count(array_filter($checks));
                $score = ($checks_passed / count($checks)) * 100;
                $result = [
                    'name' => 'Session-Konfiguration',
                    'status' => $score >= 70 ? 'success' : 'warning',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => round($score),
                    'details' => "$checks_passed von " . count($checks) . " Sicherheitsprüfungen bestanden",
                    'color' => $score >= 70 ? 'success' : 'warning'
                ];
            } else {
                $session_save_path = session_save_path();
                $session_cookie_httponly = ini_get('session.cookie_httponly') == '1';
                $session_cookie_secure = ini_get('session.cookie_secure') == '1' || !isset($_SERVER['HTTPS']);
                $score = 0;
                if ($session_cookie_httponly) $score += 50;
                if ($session_cookie_secure) $score += 50;
                $result = [
                    'name' => 'Session-Konfiguration',
                    'status' => $score >= 70 ? 'success' : 'warning',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => $score,
                    'details' => ($session_cookie_secure && $session_cookie_httponly) ? 'Session sicher konfiguriert' : 'Session-Sicherheit kann verbessert werden',
                    'color' => $score >= 70 ? 'success' : 'warning'
                ];
            }
            break;
            
        case 'p2_api':
            $p2_api_exists = file_exists('../includes/p2_api.php');
            if ($p2_api_exists) {
                require_once '../includes/p2_api.php';
                $p2api = new P2API();
                $test_result = $p2api->apiCall('/health', 'GET', [], 2, 0);
                $connection_ok = isset($test_result['success']) || isset($test_result['data']);
                $result = [
                    'name' => 'P2 API System',
                    'status' => $connection_ok ? 'success' : 'warning',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => $connection_ok ? 100 : 50,
                    'details' => $connection_ok ? 'P2 API verfügbar und erreichbar' : 'P2 API verfügbar, Verbindung fehlgeschlagen',
                    'color' => $connection_ok ? 'success' : 'warning'
                ];
            } else {
                $result = [
                    'name' => 'P2 API System',
                    'status' => 'error',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => 0,
                    'details' => 'P2 API nicht gefunden',
                    'color' => 'error'
                ];
            }
            break;
            
        case 'resy_hybrid_api':
            $resy_api_exists = file_exists('../../api/resy_data_extraction_breakthrough.php');
            if ($resy_api_exists) {
                $api_content = file_get_contents('../../api/resy_data_extraction_breakthrough.php');
                $features = [
                    'validateAndUpdateCookies' => strpos($api_content, 'function validateAndUpdateCookies') !== false,
                    'performAutomaticCookieManagement' => strpos($api_content, 'function performAutomaticCookieManagement') !== false,
                    'hybrid_approach' => strpos($api_content, 'Hybrid-Ansatz') !== false
                ];
                $features_found = count(array_filter($features));
                $score = ($features_found / count($features)) * 100;
                $result = [
                    'name' => 'RESY Hybrid API',
                    'status' => $score >= 70 ? 'success' : 'warning',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => round($score),
                    'details' => "$features_found von " . count($features) . " Hybrid-Features gefunden",
                    'color' => $score >= 70 ? 'success' : 'warning'
                ];
            } else {
                $result = [
                    'name' => 'RESY Hybrid API',
                    'status' => 'error',
                    'time' => (microtime(true) - $start) * 1000,
                    'score' => 0,
                    'details' => 'RESY API nicht gefunden',
                    'color' => 'error'
                ];
            }
            break;
            
        default:
            ob_clean();
            echo json_encode(['success' => false, 'error' => 'Unbekannter Test: ' . $test_key]);
            ob_end_flush();
            exit;
    }
    
    // Alle Ausgaben löschen und nur JSON senden
    ob_clean();
    echo json_encode([
        'success' => true,
        'result' => $result,
        'test_key' => $test_key
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    ob_end_flush();
    
} catch (Exception $e) {
    ob_clean();
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'test_key' => $test_key ?? 'unknown'
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    ob_end_flush();
} catch (Error $e) {
    ob_clean();
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage(),
        'test_key' => $test_key ?? 'unknown'
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    ob_end_flush();
}
?>

