<?php
/**
 * Projekt-Server API Proxy
 * Vermittelt zwischen Frontend und dem Tailscale-Projekt-Server (100.82.207.72:5001)
 * Sichert die Kommunikation ab und stellt Daten für das Frontend bereit
 */

header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

session_start();

// Prüfe Admin-Authentifizierung
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

// Projekt-Server Konfiguration laden
$configFile = dirname(__DIR__) . '/config/project_server_config.php';
if (file_exists($configFile)) {
    $projectServerConfig = require $configFile;
} else {
    // Fallback-Konfiguration
    $projectServerConfig = [
        'host' => '100.82.207.72',
        'port' => '5001',
        'timeout' => 30,
        'connect_timeout' => 10,
        'ssl_verify' => false,
        'default_endpoints' => [
            'health' => 'health',
            'status' => 'status',
            'info' => 'info',
            'api' => 'api'
        ]
    ];
}

define('PROJECT_SERVER_HOST', $projectServerConfig['host']);
define('PROJECT_SERVER_PORT', $projectServerConfig['port']);
define('PROJECT_SERVER_URL', 'http://' . PROJECT_SERVER_HOST . ':' . PROJECT_SERVER_PORT);
define('REQUEST_TIMEOUT', $projectServerConfig['timeout']);
define('CONNECT_TIMEOUT', $projectServerConfig['connect_timeout']);
define('SSL_VERIFY', $projectServerConfig['ssl_verify']);

/**
 * Führt einen API-Request an den Projekt-Server aus
 */
function makeProjectServerRequest($endpoint, $method = 'GET', $data = null, $headers = []) {
    $url = PROJECT_SERVER_URL . '/' . ltrim($endpoint, '/');
    
    // Standard-Headers
    $defaultHeaders = [
        'Content-Type: application/json',
        'Accept: application/json'
    ];
    
    $requestHeaders = array_merge($defaultHeaders, $headers);
    
    // cURL initialisieren
    $ch = curl_init();
    
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => REQUEST_TIMEOUT,
        CURLOPT_CONNECTTIMEOUT => CONNECT_TIMEOUT,
        CURLOPT_HTTPHEADER => $requestHeaders,
        CURLOPT_SSL_VERIFYPEER => SSL_VERIFY,
        CURLOPT_SSL_VERIFYHOST => SSL_VERIFY ? 2 : false,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_MAXREDIRS => 3
    ]);
    
    // Methode-spezifische Optionen
    switch (strtoupper($method)) {
        case 'POST':
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data !== null) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, is_string($data) ? $data : json_encode($data));
            }
            break;
            
        case 'PUT':
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
            if ($data !== null) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, is_string($data) ? $data : json_encode($data));
            }
            break;
            
        case 'DELETE':
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
            break;
            
        case 'GET':
        default:
            // GET ist Standard
            break;
    }
    
    // Request ausführen
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    $errno = curl_errno($ch);
    
    curl_close($ch);
    
    // Fehlerbehandlung
    if ($errno !== 0) {
        return [
            'success' => false,
            'error' => 'Verbindungsfehler: ' . $error,
            'http_code' => 0
        ];
    }
    
    // Response parsen
    $decodedResponse = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        // Wenn kein JSON, rohe Response zurückgeben
        return [
            'success' => $httpCode >= 200 && $httpCode < 300,
            'data' => $response,
            'http_code' => $httpCode,
            'raw' => true
        ];
    }
    
    return [
        'success' => $httpCode >= 200 && $httpCode < 300,
        'data' => $decodedResponse,
        'http_code' => $httpCode
    ];
}

// Hauptlogik
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? 'get';
$endpoint = $_GET['endpoint'] ?? '';

// Request-Body für POST/PUT lesen
$requestData = null;
if (in_array($method, ['POST', 'PUT'])) {
    $rawInput = file_get_contents('php://input');
    if (!empty($rawInput)) {
        $requestData = json_decode($rawInput, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $requestData = $rawInput; // Falls kein JSON, roh verwenden
        }
    }
}

try {
    // Endpoint validieren
    if (empty($endpoint)) {
        // Wenn kein Endpoint, versuche Standard-Endpoints aus Konfiguration
        $defaultEndpoints = $projectServerConfig['default_endpoints'] ?? [];
        if (isset($defaultEndpoints[$action])) {
            $endpoint = $defaultEndpoints[$action];
        } else {
            $endpoint = $defaultEndpoints['api'] ?? 'api'; // Fallback
        }
    }
    
    // Query-Parameter sammeln (außer endpoint und action)
    $queryParams = [];
    foreach ($_GET as $key => $value) {
        if ($key !== 'endpoint' && $key !== 'action') {
            $queryParams[$key] = $value;
        }
    }
    
    // Query-String an Endpoint anhängen
    if (!empty($queryParams)) {
        $queryString = http_build_query($queryParams);
        $endpoint .= (strpos($endpoint, '?') === false ? '?' : '&') . $queryString;
    }
    
    // Request an Projekt-Server senden
    $result = makeProjectServerRequest($endpoint, $method, $requestData);
    
    // HTTP Status Code setzen
    http_response_code($result['http_code'] ?: 500);
    
    // Response zurückgeben
    echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Server-Fehler: ' . $e->getMessage()
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
}
