<?php
/**
 * API: Lade Templates von HelloMateo
 */

// Session sicher starten (nur wenn noch nicht gestartet)
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Output-Buffering aktivieren, um sicherzustellen, dass keine Fehler das JSON zerstören
ob_start();

require_once __DIR__ . '/../../config/database.php';
require_once __DIR__ . '/../includes/hellomateo_api.php';

// Funktion zum Extrahieren von Text aus Slate-Content (HelloMateo Format)
function extractTextFromSlate($content) {
    if (!is_array($content) || empty($content)) {
        return '';
    }
    
    $text = '';
    
    function processNode($node) {
        $result = '';
        
        if (!is_array($node)) {
            return '';
        }
        
        // Text-Node
        if (isset($node['type']) && $node['type'] === 'text' && isset($node['content'])) {
            $result .= $node['content'];
        }
        // Handlebar-Node (Platzhalter)
        elseif (isset($node['type']) && $node['type'] === 'handlebar' && isset($node['key'])) {
            $result .= '{{' . $node['key'] . '}}';
        }
        // Newline
        elseif (isset($node['type']) && $node['type'] === 'newline') {
            $result .= "\n";
        }
        // Paragraph oder andere Container mit children
        elseif (isset($node['children']) && is_array($node['children'])) {
            foreach ($node['children'] as $child) {
                $result .= processNode($child);
            }
            // Füge Zeilenumbruch nach Paragraph hinzu
            if (isset($node['type']) && $node['type'] === 'paragraph') {
                $result .= "\n";
            }
        }
        
        return $result;
    }
    
    // Starte mit root oder direkt mit content
    if (isset($content['type']) && $content['type'] === 'root' && isset($content['children'])) {
        foreach ($content['children'] as $child) {
            $text .= processNode($child);
        }
    } else {
        $text = processNode($content);
    }
    
    return trim($text);
}

// Setze Header nach require_once, damit keine Fehler das JSON zerstören
header('Content-Type: application/json');

// Prüfe Authentifizierung
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht authentifiziert']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    echo json_encode(['success' => false, 'error' => 'Nur GET erlaubt']);
    exit;
}

$autohausId = $_GET['autohaus_id'] ?? null;

if (!$autohausId) {
    echo json_encode(['success' => false, 'error' => 'Autohaus-ID erforderlich']);
    exit;
}

try {
    // Prüfe zuerst ob API-Key in Datenbank existiert
    $db = new Database();
    $conn = $db->getConnection();
    $stmt = $conn->prepare("SELECT id, name, whatsapp_api_key FROM autohaus WHERE id = ?");
    $stmt->execute([$autohausId]);
    $autohaus = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$autohaus) {
        echo json_encode(['success' => false, 'error' => 'Autohaus nicht gefunden']);
        exit;
    }
    
    if (empty($autohaus['whatsapp_api_key']) || trim($autohaus['whatsapp_api_key']) === '') {
        echo json_encode([
            'success' => false, 
            'error' => 'WhatsApp API-Key für dieses Autohaus nicht konfiguriert. Bitte konfigurieren Sie den API-Key in den Einstellungen.',
            'autohaus_name' => $autohaus['name']
        ]);
        exit;
    }
    
    // Erstelle HelloMateo API Instanz
    // WICHTIG: Konstruktor erwartet (apiKey, autohausId) - nicht (autohausId)
    // WICHTIG: autohausId muss als Integer übergeben werden
    $autohausIdInt = intval($autohausId);
    $hellomateo = new HelloMateoAPI(null, $autohausIdInt);
    
    // Prüfe ob API-Key geladen wurde
    $apiKeyInfo = $hellomateo->getApiKeyInfo();
    if (!$apiKeyInfo['has_key']) {
        echo json_encode([
            'success' => false, 
            'error' => 'API-Key konnte nicht geladen werden. Bitte überprüfen Sie die Konfiguration.',
            'api_key_info' => $apiKeyInfo,
            'autohaus_id_passed' => $autohausIdInt,
            'autohaus_id_original' => $autohausId
        ]);
        exit;
    }
    
    // Debug: Logge API-Key Info (ohne vollständigen Key)
    error_log("load_hellomateo_templates.php: API-Key Info - has_key: " . ($apiKeyInfo['has_key'] ? 'true' : 'false') . ", key_length: " . ($apiKeyInfo['key_length'] ?? 'N/A'));
    
    // Lade alle Templates (mit hohem Limit)
    $result = $hellomateo->listTemplates(1000, 0);
    
    // Debug: Logge Ergebnis
    error_log("load_hellomateo_templates.php: listTemplates Result - success: " . ($result['success'] ? 'true' : 'false') . ", http_code: " . ($result['http_code'] ?? 'N/A') . ", error: " . ($result['error'] ?? 'N/A'));
    
    // Debug: Logge erste Template-Struktur für Analyse
    if ($result['success'] && is_array($result['data']) && !empty($result['data'])) {
        $firstTemplate = $result['data'][0];
        error_log("load_hellomateo_templates.php: First Template Structure - Keys: " . implode(', ', array_keys($firstTemplate)));
        error_log("load_hellomateo_templates.php: First Template - Has components: " . (isset($firstTemplate['components']) ? 'YES' : 'NO'));
        if (isset($firstTemplate['components'])) {
            error_log("load_hellomateo_templates.php: First Template - Components type: " . gettype($firstTemplate['components']) . ", Count: " . (is_array($firstTemplate['components']) ? count($firstTemplate['components']) : 'N/A'));
        }
    }
    
    if ($result['success'] && is_array($result['data'])) {
        // Formatiere Templates für Frontend
        // WICHTIG: HelloMateo API gibt Components möglicherweise nicht vollständig in der Liste zurück
        // Wir müssen für jedes Template die vollständigen Details laden, um Body/Header zu bekommen
        $templates = [];
        
        foreach ($result['data'] as $template) {
            // WICHTIG: Lade NICHT für jedes Template die Details einzeln
            // Das führt zu Rate Limiting (HTTP 429)
            // Verwende die Daten aus der Liste und lade Details nur bei Bedarf (lazy loading)
            $fullTemplateData = $template;
            // Status normalisieren - HelloMateo API kann verschiedene Status-Formate haben
            // Prüfe verschiedene mögliche Status-Felder
            $status = null;
            
            // Versuche verschiedene Status-Felder
            // HelloMateo API kann Status in verschiedenen Feldern haben
            if (isset($template['status']) && !empty($template['status'])) {
                $status = $template['status'];
            } elseif (isset($template['approval_status']) && !empty($template['approval_status'])) {
                $status = $template['approval_status'];
            } elseif (isset($template['meta_status']) && !empty($template['meta_status'])) {
                $status = $template['meta_status'];
            } elseif (isset($template['state']) && !empty($template['state'])) {
                $status = $template['state'];
            } elseif (isset($template['request_approvals'])) {
                // Falls request_approvals vorhanden ist, aber kein Status, verwende PENDING
                // request_approvals: true bedeutet, dass Template noch genehmigt werden muss
                $status = $template['request_approvals'] ? 'PENDING' : 'APPROVED';
            }
            
            // Normalisiere Status zu UPPERCASE
            if ($status) {
                $status = strtoupper(trim($status));
                // Korrigiere häufige Varianten
                if ($status === 'PENDING' || $status === 'AWAITING_APPROVAL') {
                    $status = 'PENDING';
                } elseif ($status === 'APPROVED' || $status === 'ACCEPTED' || $status === 'ACTIVE') {
                    $status = 'APPROVED';
                } elseif ($status === 'REJECTED' || $status === 'DECLINED') {
                    $status = 'REJECTED';
                }
            } else {
                $status = 'UNKNOWN';
            }
            
            // Components extrahieren - HelloMateo API verwendet content (Slate-Format) und text, nicht components!
            // WICHTIG: Die API gibt "content" (Slate-Format) und "text" (String) zurück, nicht "components"
            $components = [];
            
            // Prüfe zuerst ob Components direkt vorhanden sind (falls vorhanden)
            if (isset($template['components']) && is_array($template['components']) && !empty($template['components'])) {
                $components = $template['components'];
                error_log("Template Components gefunden - Count: " . count($components));
            } else {
                // HelloMateo API Format: content (Slate) + text + footer
                // Baue Components aus content/text/footer
                
                // BODY aus text oder content extrahieren
                $bodyText = '';
                if (isset($template['text']) && !empty($template['text'])) {
                    $bodyText = $template['text'];
                } elseif (isset($template['content']) && is_array($template['content'])) {
                    // Extrahiere Text aus Slate-Format
                    $bodyText = extractTextFromSlate($template['content']);
                }
                
                if (!empty($bodyText)) {
                    $components[] = [
                        'type' => 'BODY',
                        'text' => $bodyText,
                        'content' => $template['content'] ?? null
                    ];
                    error_log("Template Body aus text/content extrahiert - Länge: " . strlen($bodyText));
                }
                
                // HEADER aus header_media_filename prüfen
                if (isset($template['header_media_filename']) && !empty($template['header_media_filename'])) {
                    // Bestimme Format basierend auf Dateierweiterung
                    $filename = $template['header_media_filename'];
                    $format = 'IMAGE';
                    if (preg_match('/\.(mp4|avi|mov)$/i', $filename)) {
                        $format = 'VIDEO';
                    } elseif (preg_match('/\.(pdf|doc|docx)$/i', $filename)) {
                        $format = 'DOCUMENT';
                    }
                    
                    // Versuche URL zu konstruieren (HelloMateo könnte Bilder über eine URL bereitstellen)
                    $imageUrl = null;
                    if (isset($template['header_media_url'])) {
                        $imageUrl = $template['header_media_url'];
                    } elseif (isset($template['header_media'])) {
                        $imageUrl = is_string($template['header_media']) ? $template['header_media'] : ($template['header_media']['url'] ?? null);
                    }
                    
                    $components[] = [
                        'type' => 'HEADER',
                        'format' => $format,
                        'example' => $filename,
                        'filename' => $filename,
                        'url' => $imageUrl
                    ];
                    error_log("Template Header gefunden - Format: " . $format . ", Filename: " . $filename . ", URL: " . ($imageUrl ?? 'N/A'));
                }
                
                // FOOTER
                if (isset($template['footer']) && !empty($template['footer'])) {
                    $components[] = [
                        'type' => 'FOOTER',
                        'text' => $template['footer']
                    ];
                    error_log("Template Footer gefunden - Text: " . $template['footer']);
                }
                
                // BUTTONS
                if (isset($template['buttons']) && is_array($template['buttons']) && !empty($template['buttons'])) {
                    $components[] = [
                        'type' => 'BUTTONS',
                        'buttons' => $template['buttons']
                    ];
                    error_log("Template Buttons gefunden - Count: " . count($template['buttons']));
                }
            }
            
            // Fallback: Prüfe ob body/header/footer direkt vorhanden sind (für Kompatibilität)
            if (empty($components) && (isset($template['body']) || isset($template['header']) || isset($template['footer']))) {
                if (isset($template['header'])) {
                    $headerText = is_array($template['header']) ? ($template['header']['text'] ?? $template['header']['content'] ?? '') : $template['header'];
                    $components[] = [
                        'type' => 'HEADER',
                        'text' => $headerText,
                        'format' => is_array($template['header']) ? ($template['header']['format'] ?? null) : null
                    ];
                }
                if (isset($template['body'])) {
                    $bodyText = '';
                    if (is_array($template['body'])) {
                        $bodyText = $template['body']['text'] ?? $template['body']['content'] ?? $template['body']['message'] ?? '';
                    } else {
                        $bodyText = $template['body'];
                    }
                    $components[] = [
                        'type' => 'BODY',
                        'text' => $bodyText,
                        'content' => is_array($template['body']) ? ($template['body']['content'] ?? null) : null
                    ];
                }
                if (isset($template['footer'])) {
                    $footerText = is_array($template['footer']) ? ($template['footer']['text'] ?? $template['footer']['content'] ?? '') : $template['footer'];
                    $components[] = [
                        'type' => 'FOOTER',
                        'text' => $footerText
                    ];
                }
            }
            
            // WICHTIG: Components werden aus der Liste extrahiert
            // Falls BODY/HEADER fehlen, werden sie bei Bedarf geladen (lazy loading im Frontend)
            
            // Falls immer noch keine Components, logge vollständige Template-Struktur
            if (empty($components)) {
                error_log("Template Components Debug - Keine Components gefunden. Template Keys: " . implode(', ', array_keys($template)));
                error_log("Template Components Debug - Template Sample (erste 2000 Zeichen): " . substr(json_encode($template, JSON_PRETTY_PRINT), 0, 2000));
            } else {
                // Debug-Logging für Components
                error_log("Template Debug - ID: " . ($template['id'] ?? 'N/A') . ", Status: " . $status . ", Components Count: " . count($components));
                
                // Logge alle Components
                foreach ($components as $idx => $comp) {
                    $compType = $comp['type'] ?? 'UNKNOWN';
                    $hasText = isset($comp['text']) && !empty($comp['text']);
                    $hasContent = isset($comp['content']);
                    error_log("Template Component #{$idx}: type={$compType}, has_text=" . ($hasText ? 'YES' : 'NO') . ", text_length=" . ($hasText ? strlen($comp['text']) : 0));
                }
            }
            
            // Debug-Logging für Status und Components
            error_log("Template Debug - ID: " . ($template['id'] ?? 'N/A') . ", Status: " . $status . ", Components Count: " . count($components));
            if (!empty($components)) {
                // Logge Body-Component Details
                $bodyComponent = array_filter($components, function($c) { return isset($c['type']) && $c['type'] === 'BODY'; });
                if (!empty($bodyComponent)) {
                    $body = reset($bodyComponent);
                    error_log("Template Body Debug - Text: " . ($body['text'] ?? 'NULL') . ", Content: " . (isset($body['content']) ? 'PRESENT' : 'NULL'));
                } else {
                    error_log("Template Body Debug - Kein BODY Component gefunden");
                }
            }
            
            // Prüfe ob Body/Header/Footer direkt im Template sind (nicht in Components)
            // HelloMateo API kann diese Felder direkt im Template-Objekt haben
            $bodyDirect = $template['body'] ?? null;
            $headerDirect = $template['header'] ?? null;
            $footerDirect = $template['footer'] ?? null;
            
            // Füge direkte Felder zu Components hinzu, falls sie nicht bereits vorhanden sind
            if ($bodyDirect && !array_filter($components, function($c) { return isset($c['type']) && $c['type'] === 'BODY'; })) {
                $bodyText = is_array($bodyDirect) ? ($bodyDirect['text'] ?? $bodyDirect['content'] ?? '') : $bodyDirect;
                if ($bodyText) {
                    $components[] = [
                        'type' => 'BODY',
                        'text' => $bodyText,
                        'content' => is_array($bodyDirect) ? ($bodyDirect['content'] ?? null) : null
                    ];
                    error_log("Template Debug - ID: " . ($template['id'] ?? 'N/A') . " - Body direkt aus Template hinzugefügt");
                }
            }
            
            if ($headerDirect && !array_filter($components, function($c) { return isset($c['type']) && $c['type'] === 'HEADER'; })) {
                $headerText = is_array($headerDirect) ? ($headerDirect['text'] ?? $headerDirect['content'] ?? '') : $headerDirect;
                if ($headerText) {
                    $components[] = [
                        'type' => 'HEADER',
                        'text' => $headerText,
                        'format' => is_array($headerDirect) ? ($headerDirect['format'] ?? null) : null
                    ];
                    error_log("Template Debug - ID: " . ($template['id'] ?? 'N/A') . " - Header direkt aus Template hinzugefügt");
                }
            }
            
            if ($footerDirect && !array_filter($components, function($c) { return isset($c['type']) && $c['type'] === 'FOOTER'; })) {
                $footerText = is_array($footerDirect) ? ($footerDirect['text'] ?? $footerDirect['content'] ?? '') : $footerDirect;
                if ($footerText) {
                    $components[] = [
                        'type' => 'FOOTER',
                        'text' => $footerText
                    ];
                    error_log("Template Debug - ID: " . ($template['id'] ?? 'N/A') . " - Footer direkt aus Template hinzugefügt");
                }
            }
            
            // Füge auch direkte Felder zum Template-Objekt hinzu (für Frontend-Fallback)
            $templateData = [
                'id' => $template['id'] ?? '',
                'name' => $template['name'] ?? $template['id'] ?? 'Unbenannt',
                'status' => $status,
                'language' => $template['language'] ?? 'de',
                'category' => $template['category'] ?? 'UTILITY',
                'components' => $components,
                'created_at' => $template['created_at'] ?? null,
                'updated_at' => $template['updated_at'] ?? null,
            ];
            
            // Füge direkte Felder hinzu (falls vorhanden)
            if ($bodyDirect) {
                $templateData['body'] = $bodyDirect;
            }
            if ($headerDirect) {
                $templateData['header'] = $headerDirect;
            }
            if ($footerDirect) {
                $templateData['footer'] = $footerDirect;
            }
            
            $templates[] = $templateData;
        }
        
        // Lösche Output-Buffer vor JSON-Ausgabe
        ob_clean();
        
        echo json_encode([
            'success' => true,
            'data' => $templates,
            'count' => count($templates)
        ]);
    } else {
        // Detaillierte Fehlerbehandlung
        $errorMessage = $result['error'] ?? 'Templates konnten nicht geladen werden';
        
        // Prüfe ob es ein HTTP-Fehler ist
        $httpCode = $result['http_code'] ?? null;
        if ($httpCode === 401 || $httpCode === 403) {
            $errorMessage = 'Invalid token - Der API-Key ist ungültig oder abgelaufen. Bitte überprüfen Sie den API-Key in den Einstellungen.';
        }
        
        error_log("load_hellomateo_templates.php: Fehler beim Laden der Templates - " . $errorMessage . " (HTTP: " . ($httpCode ?? 'N/A') . ")");
        
        // Lösche Output-Buffer vor JSON-Ausgabe
        ob_clean();
        
        echo json_encode([
            'success' => false,
            'error' => $errorMessage,
            'http_code' => $httpCode,
            'data' => []
        ]);
    }
} catch (Exception $e) {
    error_log("load_hellomateo_templates.php: Exception - " . $e->getMessage() . " in " . $e->getFile() . ":" . $e->getLine());
    
    // Lösche Output-Buffer vor JSON-Ausgabe
    ob_clean();
    
    echo json_encode([
        'success' => false,
        'error' => 'Fehler: ' . $e->getMessage(),
        'data' => []
    ]);
}

// Beende Output-Buffering
ob_end_flush();


