<?php
/**
 * Excel Export für Termine
 * Erstellt eine professionelle Excel-Datei im Format der alten Tabelle
 */

session_start();
require_once '../../config/database.php';

// Nur für eingeloggte Admins
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    die('Nicht autorisiert');
}

$db = new Database();
$conn = $db->getConnection();

// Parameter aus GET/POST
$event_id = $_GET['event_id'] ?? $_POST['event_id'] ?? null;
$appointment_ids = $_POST['appointment_ids'] ?? null; // Für Bulk-Export
$autohaus_id = $_SESSION['autohaus_id'] ?? null;
$user_role = $_SESSION['user_role'] ?? 'admin';
$can_switch_autohaus = $_SESSION['can_switch_autohaus'] ?? false;

// Hole Termine
if ($appointment_ids && is_array($appointment_ids)) {
    // Bulk-Export: Nur ausgewählte Termine
    $placeholders = implode(',', array_fill(0, count($appointment_ids), '?'));
    $sql = "
        SELECT a.*, a.user_id, e.name as event_name, e.event_date, e.max_monteure,
               slot_stats.booked_count as booked_appointments,
               slot_stats.total_count as total_appointments,
               ah.name as autohaus_name,
               au.username as approved_by_username
        FROM appointments a
        JOIN events e ON a.event_id = e.id
        LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
        LEFT JOIN admin_users au ON a.approved_by = au.id
        LEFT JOIN (
            SELECT 
                event_id, 
                slot_date, 
                slot_time,
                COUNT(*) as total_count,
                COUNT(CASE WHEN status IN ('booked', 'vorbereitet', 'completed') THEN 1 END) as booked_count
            FROM appointments 
            GROUP BY event_id, slot_date, slot_time
        ) slot_stats ON a.event_id = slot_stats.event_id 
                     AND a.slot_date = slot_stats.slot_date 
                     AND a.slot_time = slot_stats.slot_time
        WHERE a.id IN ($placeholders)
        ORDER BY a.slot_date ASC, a.slot_time ASC
    ";
    $stmt = $conn->prepare($sql);
    $stmt->execute($appointment_ids);
} else {
    // Alle Termine oder gefiltert nach Event
    $sql = "
        SELECT a.*, a.user_id, e.name as event_name, e.event_date, e.max_monteure,
               slot_stats.booked_count as booked_appointments,
               slot_stats.total_count as total_appointments,
               ah.name as autohaus_name,
               au.username as approved_by_username
        FROM appointments a
        JOIN events e ON a.event_id = e.id
        LEFT JOIN autohaus ah ON a.autohaus_id = ah.id
        LEFT JOIN admin_users au ON a.approved_by = au.id
        LEFT JOIN (
            SELECT 
                event_id, 
                slot_date, 
                slot_time,
                COUNT(*) as total_count,
                COUNT(CASE WHEN status IN ('booked', 'vorbereitet', 'completed') THEN 1 END) as booked_count
            FROM appointments 
            GROUP BY event_id, slot_date, slot_time
        ) slot_stats ON a.event_id = slot_stats.event_id 
                     AND a.slot_date = slot_stats.slot_date 
                     AND a.slot_time = slot_stats.slot_time
        WHERE 1=1
    ";
    
    $params = [];
    
    if ($event_id) {
        $sql .= " AND a.event_id = ?";
        $params[] = $event_id;
    }
    
    // Filter nach Autohaus wenn nötig
    if ($user_role !== 'admin' || !$can_switch_autohaus) {
        if ($autohaus_id) {
            $sql .= " AND a.autohaus_id = ?";
            $params[] = $autohaus_id;
        } else {
            $sql .= " AND a.autohaus_id IS NULL";
        }
    }
    
    $sql .= " ORDER BY a.slot_date ASC, a.slot_time ASC";
    
    $stmt = $conn->prepare($sql);
    $stmt->execute($params);
}

$appointments = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Excel-XML generieren
header('Content-Type: application/vnd.ms-excel; charset=utf-8');
header('Content-Disposition: attachment; filename="Termine_Export_' . date('Y-m-d') . '.xls"');
header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
header('Pragma: public');

// Excel XML Header
echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
echo '<?mso-application progid="Excel.Sheet"?>' . "\n";
echo '<Workbook xmlns="urn:schemas-microsoft-com:office:spreadsheet"' . "\n";
echo ' xmlns:o="urn:schemas-microsoft-com:office:office"' . "\n";
echo ' xmlns:x="urn:schemas-microsoft-com:office:excel"' . "\n";
echo ' xmlns:ss="urn:schemas-microsoft-com:office:spreadsheet"' . "\n";
echo ' xmlns:html="http://www.w3.org/TR/REC-html40">' . "\n";

// Styles
echo '<Styles>' . "\n";
echo '<Style ss:ID="Header">' . "\n";
echo '<Font ss:Bold="1" ss:Size="11" ss:Color="#FFFFFF"/>' . "\n";
echo '<Interior ss:Color="#003057" ss:Pattern="Solid"/>' . "\n";
echo '<Alignment ss:Horizontal="Center" ss:Vertical="Center" ss:WrapText="1"/>' . "\n";
echo '<Borders>' . "\n";
echo '<Border ss:Position="Bottom" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#000000"/>' . "\n";
echo '<Border ss:Position="Top" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#000000"/>' . "\n";
echo '<Border ss:Position="Left" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#000000"/>' . "\n";
echo '<Border ss:Position="Right" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#000000"/>' . "\n";
echo '</Borders>' . "\n";
echo '</Style>' . "\n";
echo '<Style ss:ID="EvenRow">' . "\n";
echo '<Interior ss:Color="#F8F9FA" ss:Pattern="Solid"/>' . "\n";
echo '<Borders>' . "\n";
echo '<Border ss:Position="Bottom" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '<Border ss:Position="Top" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '<Border ss:Position="Left" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '<Border ss:Position="Right" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '</Borders>' . "\n";
echo '<Alignment ss:Vertical="Center" ss:WrapText="1"/>' . "\n";
echo '</Style>' . "\n";
echo '<Style ss:ID="OddRow">' . "\n";
echo '<Interior ss:Color="#FFFFFF" ss:Pattern="Solid"/>' . "\n";
echo '<Borders>' . "\n";
echo '<Border ss:Position="Bottom" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '<Border ss:Position="Top" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '<Border ss:Position="Left" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '<Border ss:Position="Right" ss:LineStyle="Continuous" ss:Weight="1" ss:Color="#E0E0E0"/>' . "\n";
echo '</Borders>' . "\n";
echo '<Alignment ss:Vertical="Center" ss:WrapText="1"/>' . "\n";
echo '</Style>' . "\n";
echo '</Styles>' . "\n";

// Worksheet
echo '<Worksheet ss:Name="Termine">' . "\n";
echo '<Table>' . "\n";

// Spaltenbreiten
echo '<Column ss:Width="80"/>' . "\n";  // Uhrzeit
echo '<Column ss:Width="150"/>' . "\n"; // Kunde
echo '<Column ss:Width="120"/>' . "\n"; // Kennzeichen
echo '<Column ss:Width="80"/>' . "\n";  // eingelagert
echo '<Column ss:Width="120"/>' . "\n"; // Einlagerungsnr.
echo '<Column ss:Width="100"/>' . "\n"; // bringt der Kd. mit
echo '<Column ss:Width="120"/>' . "\n"; // WR Fzg. oder Einl.
echo '<Column ss:Width="100"/>' . "\n"; // bestellt zum
echo '<Column ss:Width="100"/>' . "\n"; // Auftrag
echo '<Column ss:Width="180"/>' . "\n"; // Wintercheck/frühjahrescheck
echo '<Column ss:Width="120"/>' . "\n"; // Auftragsnummer
echo '<Column ss:Width="120"/>' . "\n"; // Telefonnummer
echo '<Column ss:Width="250"/>' . "\n"; // Informationen

// Kopfzeile
echo '<Row ss:Height="25">' . "\n";
$headers = [
    'Uhrzeit',
    'Kunde',
    'Kennzeichen',
    'eingelagert',
    'Einlagerungsnr.',
    'bringt der Kd. mit',
    'WR Fzg. oder Einl.',
    'bestellt zum',
    'Auftrag',
    'Wintercheck/frühjahrescheck J/N',
    'Auftragsnummer',
    'Telefonnummer',
    'Informationen'
];

foreach ($headers as $header) {
    echo '<Cell ss:StyleID="Header"><Data ss:Type="String">' . htmlspecialchars($header, ENT_XML1) . '</Data></Cell>' . "\n";
}
echo '</Row>' . "\n";

// Datenzeilen
foreach ($appointments as $index => $apt) {
    $rowStyle = ($index % 2 === 1) ? 'EvenRow' : 'OddRow';
    
    // Uhrzeit
    $time = !empty($apt['slot_time']) ? date('H:i', strtotime($apt['slot_time'])) : '';
    
    // Kunde
    $customer = htmlspecialchars($apt['customer_name'] ?? '', ENT_XML1);
    
    // Kennzeichen
    $license_plate = htmlspecialchars($apt['license_plate'] ?? '', ENT_XML1);
    
    // eingelagert (x wenn Kunde keine Räder mitbringt, also wenn brings_wheels = 0 oder false)
    $brings_wheels_value = $apt['brings_wheels'] ?? null;
    $stored = ($brings_wheels_value === 0 || $brings_wheels_value === false || $brings_wheels_value === '0') ? 'x' : '';
    
    // Einlagerungsnummer
    $storage_number = htmlspecialchars($apt['storage_number'] ?? '', ENT_XML1);
    
    // bringt der Kd. mit
    $brings_wheels = ($brings_wheels_value === 1 || $brings_wheels_value === true || $brings_wheels_value === '1') ? 'Ja' : 'Nein';
    
    // WR Fzg. oder Einl. (Winterreifen Fahrzeug oder Einlagerung)
    $wr_type = '';
    if ($brings_wheels_value !== null) {
        $wr_type = ($brings_wheels_value === 1 || $brings_wheels_value === true || $brings_wheels_value === '1') ? 'Fzg.' : 'Einl.';
    }
    
    // bestellt zum (tire_order_date)
    $ordered_for = '';
    if (!empty($apt['tire_order_date'])) {
        $ordered_for = date('d.m.Y', strtotime($apt['tire_order_date']));
    }
    
    // Auftrag (vermutlich order_number oder status)
    $order = htmlspecialchars($apt['order_number'] ?? '', ENT_XML1);
    
    // Wintercheck/frühjahrescheck J/N
    $seasonal_check = '';
    if (!empty($apt['seasonal_check']) && $apt['seasonal_check'] !== 'none') {
        $seasonal_check = ($apt['seasonal_check'] === 'spring') ? 'Frühjahrescheck: Ja' : 'Wintercheck: Ja';
    } else {
        $seasonal_check = 'Nein';
    }
    
    // Auftragsnummer (order_number)
    $order_number = htmlspecialchars($apt['order_number'] ?? '', ENT_XML1);
    
    // Telefonnummer
    $phone = htmlspecialchars($apt['phone'] ?? '', ENT_XML1);
    
    // Informationen (info_text oder admin_notes)
    $info = '';
    if (!empty($apt['info_text'])) {
        $info = htmlspecialchars($apt['info_text'], ENT_XML1);
    }
    if (!empty($apt['admin_notes'])) {
        if ($info) $info .= ' | ';
        $info .= htmlspecialchars($apt['admin_notes'], ENT_XML1);
    }
    
    echo '<Row>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $time . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $customer . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $license_plate . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $stored . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $storage_number . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $brings_wheels . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $wr_type . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $ordered_for . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $order . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $seasonal_check . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $order_number . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $phone . '</Data></Cell>' . "\n";
    echo '<Cell ss:StyleID="' . $rowStyle . '"><Data ss:Type="String">' . $info . '</Data></Cell>' . "\n";
    echo '</Row>' . "\n";
}

echo '</Table>' . "\n";
echo '</Worksheet>' . "\n";
echo '</Workbook>' . "\n";

