<?php
/**
 * AJAX-Endpoint: Automatische Drucker-Erkennung für Autohaus
 * Prüft verfügbare Drucker, Server-Verbindung, etc.
 */

session_start();
header('Content-Type: application/json');

if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    echo json_encode(['success' => false, 'error' => 'Nicht autorisiert']);
    exit;
}

require_once __DIR__ . '/../includes/smart_printer_detection.php';

$autohausId = $_GET['autohaus_id'] ?? null;
$printer_name_local = $_GET['printer_name_local'] ?? null;
$printer_name_server = $_GET['printer_name_server'] ?? null;
$printer_server_ip = $_GET['printer_server_ip'] ?? null;
$printer_server_hostname = $_GET['printer_server_hostname'] ?? null;
$printer_server_user = $_GET['printer_server_user'] ?? null;
$printer_server_pass = $_GET['printer_server_pass'] ?? null;

$result = [
    'success' => true,
    'detection' => [
        'local_printers' => [],
        'configured_printer_found' => false,
        'configured_printer_details' => null,
        'server_connection' => [
            'reachable' => false,
            'error' => null,
            'shared_printers' => []
        ],
        'recommendations' => [],
        'errors' => []
    ]
];

try {
    // 1. Lade alle lokal installierten Drucker
    $ps_cmd = 'powershell -NoProfile -ExecutionPolicy Bypass -Command "Get-Printer | Select-Object Name, PrinterStatus, PortName, DriverName, WorkOffline | ConvertTo-Json" 2>&1';
    exec($ps_cmd, $ps_output, $ps_code);
    
    if ($ps_code === 0 && !empty($ps_output)) {
        $json_output = implode("\n", $ps_output);
        $printers = json_decode($json_output, true);
        
        if (json_last_error() === JSON_ERROR_NONE && is_array($printers)) {
            // Normalisiere Array
            if (isset($printers['Name'])) {
                $printers = [$printers];
            }
            
            foreach ($printers as $printer) {
                $name = trim($printer['Name'] ?? '');
                if (empty($name)) continue;
                
                // Filtere virtuelle Drucker (PDF, XPS, etc.)
                $virtual_printers = ['PDF', 'XPS', 'Fax', 'OneNote', 'Microsoft Print'];
                $is_virtual = false;
                foreach ($virtual_printers as $virtual) {
                    if (stripos($name, $virtual) !== false) {
                        $is_virtual = true;
                        break;
                    }
                }
                
                if (!$is_virtual) {
                    $port = trim($printer['PortName'] ?? '');
                    $status = trim($printer['PrinterStatus'] ?? '');
                    $driver = trim($printer['DriverName'] ?? '');
                    $work_offline = isset($printer['WorkOffline']) ? (bool)$printer['WorkOffline'] : false;
                    
                    // Prüfe Port-Typ
                    $port_type = 'unknown';
                    if (stripos($port, 'LPT') !== false) {
                        $port_type = 'LPT (Lokaler Port - nicht empfohlen)';
                    } elseif (stripos($port, 'COM') !== false) {
                        $port_type = 'COM (Serieller Port - nicht empfohlen)';
                    } elseif (stripos($port, '\\\\') === 0 || stripos($port, '//') === 0) {
                        $port_type = 'Netzwerk (UNC)';
                    } elseif (filter_var($port, FILTER_VALIDATE_IP) || preg_match('/^\d+\.\d+\.\d+\.\d+_\d+$/', $port)) {
                        $port_type = 'Netzwerk (IP)';
                    } else {
                        $port_type = 'Unbekannt';
                    }
                    
                    $result['detection']['local_printers'][] = [
                        'name' => $name,
                        'port' => $port,
                        'port_type' => $port_type,
                        'status' => $status,
                        'driver' => $driver,
                        'work_offline' => $work_offline,
                        'is_online' => (stripos($status, 'Normal') !== false || stripos($status, 'Ready') !== false) && !$work_offline
                    ];
                }
            }
        }
    }
    
    // 2. Prüfe, ob konfigurierter Drucker gefunden wurde
    if ($printer_name_local) {
        foreach ($result['detection']['local_printers'] as $printer) {
            if (strcasecmp($printer['name'], $printer_name_local) === 0 || 
                stripos($printer['name'], $printer_name_local) !== false) {
                $result['detection']['configured_printer_found'] = true;
                $result['detection']['configured_printer_details'] = $printer;
                break;
            }
        }
        
        if (!$result['detection']['configured_printer_found']) {
            $result['detection']['errors'][] = "Konfigurierter Drucker '$printer_name_local' wurde nicht gefunden.";
            $result['detection']['recommendations'][] = "Prüfen Sie, ob der Drucker lokal installiert ist. Verfügbare Drucker: " . implode(', ', array_column($result['detection']['local_printers'], 'name'));
        }
    }
    
    // 3. Prüfe Server-Verbindung
    if ($printer_server_ip || $printer_server_hostname) {
        $server_to_test = $printer_server_ip ?: $printer_server_hostname;
        
        // Ping-Test
        $ping_cmd = sprintf('ping -n 1 -w 1000 %s 2>&1', escapeshellarg($server_to_test));
        exec($ping_cmd, $ping_output, $ping_code);
        
        if ($ping_code === 0) {
            $result['detection']['server_connection']['reachable'] = true;
            
            // Versuche Server-Freigaben zu listen
            if ($printer_server_user && $printer_server_pass) {
                // Verbinde mit Server
                $net_use_cmd = sprintf(
                    'net use "\\\\%s" "%s" /user:"%s" 2>&1',
                    escapeshellarg($server_to_test),
                    escapeshellarg($printer_server_pass),
                    escapeshellarg($printer_server_user)
                );
                exec($net_use_cmd, $net_output, $net_code);
                
                if ($net_code === 0) {
                    // Liste Drucker-Freigaben
                    $net_view_cmd = sprintf('net view "\\\\%s" 2>&1', escapeshellarg($server_to_test));
                    exec($net_view_cmd, $view_output, $view_code);
                    
                    if ($view_code === 0) {
                        $in_printers = false;
                        foreach ($view_output as $line) {
                            if (stripos($line, 'Drucker') !== false || stripos($line, 'Printer') !== false) {
                                $in_printers = true;
                                continue;
                            }
                            
                            if ($in_printers && trim($line) && stripos($line, '---') === false) {
                                // Extrahiere Druckername
                                $parts = preg_split('/\s+/', trim($line));
                                if (!empty($parts[0])) {
                                    $printer_share = trim($parts[0]);
                                    if (!empty($printer_share)) {
                                        $result['detection']['server_connection']['shared_printers'][] = $printer_share;
                                    }
                                }
                            }
                        }
                        
                        // Prüfe, ob konfigurierter Server-Drucker gefunden wurde
                        if ($printer_name_server) {
                            $server_printer_found = false;
                            foreach ($result['detection']['server_connection']['shared_printers'] as $shared_printer) {
                                if (strcasecmp($shared_printer, $printer_name_server) === 0 || 
                                    stripos($shared_printer, $printer_name_server) !== false) {
                                    $server_printer_found = true;
                                    break;
                                }
                            }
                            
                            if (!$server_printer_found) {
                                $result['detection']['errors'][] = "Drucker '$printer_name_server' wurde auf dem Server nicht gefunden.";
                                $result['detection']['recommendations'][] = "Verfügbare Drucker auf Server: " . implode(', ', $result['detection']['server_connection']['shared_printers']);
                            }
                        }
                    }
                } else {
                    $result['detection']['server_connection']['error'] = "Verbindung zum Server fehlgeschlagen: " . implode("\n", $net_output);
                    $result['detection']['recommendations'][] = "Prüfen Sie Benutzername und Passwort für den Server-Zugriff.";
                }
            } else {
                $result['detection']['server_connection']['error'] = "Benutzername oder Passwort nicht angegeben.";
            }
        } else {
            $result['detection']['server_connection']['error'] = "Server nicht erreichbar (Ping fehlgeschlagen).";
            $result['detection']['recommendations'][] = "Prüfen Sie die Netzwerk-Verbindung und VPN-Verbindung.";
        }
    }
    
    // 4. Generiere Empfehlungen
    if (empty($result['detection']['local_printers'])) {
        $result['detection']['errors'][] = "Keine Drucker lokal installiert gefunden.";
        $result['detection']['recommendations'][] = "Installieren Sie den Drucker über Windows Explorer: \\\\$server_to_test";
    }
    
    // Prüfe Port-Typen
    $has_network_printer = false;
    foreach ($result['detection']['local_printers'] as $printer) {
        if ($printer['port_type'] === 'Netzwerk (UNC)' || $printer['port_type'] === 'Netzwerk (IP)') {
            $has_network_printer = true;
            break;
        }
    }
    
    if (!$has_network_printer && !empty($result['detection']['local_printers'])) {
        $result['detection']['recommendations'][] = "Warnung: Gefundene Drucker verwenden lokale Ports (LPT/COM). Für Netzwerk-Drucker sollten Netzwerk-Ports verwendet werden.";
    }
    
    // Prüfe Offline-Status
    foreach ($result['detection']['local_printers'] as $printer) {
        if ($printer['work_offline']) {
            $result['detection']['errors'][] = "Drucker '{$printer['name']}' ist offline.";
            $result['detection']['recommendations'][] = "Drucker '{$printer['name']}' ist offline. Aktivieren Sie 'Drucker online' in den Druckereigenschaften.";
        }
    }
    
} catch (Exception $e) {
    $result['success'] = false;
    $result['error'] = $e->getMessage();
}

echo json_encode($result, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
?>







