<?php
/**
 * Approve User API
 * Allows admins to approve users with password verification
 */

session_start();
header('Content-Type: application/json');

// Check if user is logged in as admin
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Unauthorized - Please log in as admin']);
    exit;
}

require_once __DIR__ . '/../../config/database.php';

try {
    $db = new Database();
    $conn = $db->getConnection();
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['user_id'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid input']);
        exit;
    }
    
    $user_id = intval($input['user_id']);
    $admin_notes = $input['admin_notes'] ?? '';
    $approval_reason = $input['approval_reason'] ?? '';
    $admin_password = $input['admin_password'] ?? '';
    $admin_id = $_SESSION['admin_user_id'] ?? null;
    
    // Check if approval_reason is required and provided
    if (empty($approval_reason)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Bitte geben Sie eine Begründung für die Freischaltung an.']);
        exit;
    }
    
    // Check if password is provided
    if (empty($admin_password)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Bitte geben Sie Ihr Passwort zur Bestätigung ein.']);
        exit;
    }
    
    // Verify admin password and role
    $stmt = $conn->prepare("SELECT id, password_hash, role FROM admin_users WHERE id = ?");
    $stmt->execute([$admin_id]);
    $admin = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$admin) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Admin-Benutzer nicht gefunden.']);
        exit;
    }
    
    // Check if role is admin
    if ($admin['role'] !== 'admin') {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Nur Administratoren können Benutzer freischalten.']);
        exit;
    }
    
    // Verify password
    if (!password_verify($admin_password, $admin['password_hash'])) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Falsches Passwort. Bitte versuchen Sie es erneut.']);
        exit;
    }
    
    $conn->beginTransaction();
    
    try {
        // Approve user - check if approval_reason column exists
        $checkColumn = $conn->query("SHOW COLUMNS FROM users LIKE 'approval_reason'");
        $hasApprovalReason = $checkColumn->rowCount() > 0;
        
        $sql = "UPDATE users SET status = 'approved', approved_by = ?, approved_at = NOW(), admin_notes = ?";
        $params = [$admin_id, $admin_notes];
        
        if ($hasApprovalReason) {
            $sql .= ", approval_reason = ?";
            $params[] = $approval_reason;
        } else {
            // Store in admin_notes if column doesn't exist
            $admin_notes = ($admin_notes ? $admin_notes . "\n\n" : '') . "Freischaltungsbegründung: " . $approval_reason;
            $params[1] = $admin_notes; // Update admin_notes
        }
        
        // Check for suspended columns
        $checkSuspendedBy = $conn->query("SHOW COLUMNS FROM users LIKE 'suspended_by'");
        $hasSuspendedBy = $checkSuspendedBy->rowCount() > 0;
        
        if ($hasSuspendedBy) {
            $sql .= ", suspended_by = NULL, suspended_at = NULL";
        }
        
        $sql .= ", rejected_by = NULL, rejected_at = NULL, rejection_reason = NULL WHERE id = ?";
        $params[] = $user_id;
        
        $stmt = $conn->prepare($sql);
        $stmt->execute($params);
        
        // Log the action in admin_actions table
        try {
            $tableCheck = $conn->query("SHOW TABLES LIKE 'admin_actions'");
            if ($tableCheck->rowCount() > 0) {
                $logSql = "INSERT INTO admin_actions (admin_id, user_id, action, reason, created_at) VALUES (?, ?, 'approve', ?, NOW())";
                $logStmt = $conn->prepare($logSql);
                $logReason = "Freischaltungsbegründung: " . $approval_reason . ($admin_notes ? "\n\nAdmin-Notizen: " . $admin_notes : '');
                $logStmt->execute([$admin_id, $user_id, $logReason]);
            }
        } catch (Exception $logError) {
            error_log("Error logging admin action: " . $logError->getMessage());
            // Continue even if logging fails
        }
        
        // Create notification with push
        require_once __DIR__ . '/../../includes/push_notification_helper.php';
        $result = createNotificationWithPush(
            $conn,
            $user_id,
            'Account freigeschaltet',
            'Ihr Account wurde erfolgreich freigeschaltet. Sie können sich jetzt anmelden.',
            'success',
            '/profile.php'
        );
        
        $conn->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Benutzer erfolgreich freigeschaltet'
        ]);
        
    } catch (Exception $e) {
        $conn->rollBack();
        error_log("Error approving user: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Fehler beim Freischalten: ' . $e->getMessage()]);
    }
    
} catch (Exception $e) {
    error_log("Error in approve_user API: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Internal server error']);
}
?>







